#!/usr/bin/env python3
import os, csv, io, json, pathlib
from datetime import datetime, timedelta, date
from flask import Flask, render_template, request, redirect, url_for, session, flash, send_file
from werkzeug.security import generate_password_hash, check_password_hash

try:
    import pandas as pd
except Exception:
    pd = None

APP_TITLE = "Warehouse Consumables — Lite (v4.3-shared)"
BASE_DIR = pathlib.Path(__file__).resolve().parent
DATA_DIR = BASE_DIR / "data"
USERS_FILE = DATA_DIR / "users.json"
ITEMS_FILE = DATA_DIR / "items.json"
DAILY_DIR = DATA_DIR / "logs"
AUDIT_DIR = DATA_DIR / "audit"
EXPORTS_DIR = DATA_DIR / "exports"

def ensure_storage():
    DATA_DIR.mkdir(parents=True, exist_ok=True)
    DAILY_DIR.mkdir(parents=True, exist_ok=True)
    EXPORTS_DIR.mkdir(parents=True, exist_ok=True)
    AUDIT_DIR.mkdir(parents=True, exist_ok=True)
    if not USERS_FILE.exists():
        users = {"users":[{"username":"admin","password_hash":generate_password_hash("admin"),"role":"admin"}]}
        USERS_FILE.write_text(json.dumps(users, indent=2), encoding="utf-8")
    if not ITEMS_FILE.exists():
        default_items = {"items":[
            {"name":"Bubble Wrap","unit":"meter","opening_stock":0.0,"min_stock":10.0,"barcode":"BW001"},
            {"name":"Lakban Coklat","unit":"roll","opening_stock":0.0,"min_stock":5.0,"barcode":"LC001"},
            {"name":"Lakban Bening","unit":"roll","opening_stock":0.0,"min_stock":5.0,"barcode":"LB001"},
            {"name":"Stretch Film","unit":"kg","opening_stock":0.0,"min_stock":5.0,"barcode":"SF001"},
            {"name":"Dus/Karton","unit":"pcs","opening_stock":0.0,"min_stock":20.0,"barcode":"DK001"}
        ]}
        ITEMS_FILE.write_text(json.dumps(default_items, indent=2), encoding="utf-8")

def log_audit(action, detail=""):
    ts = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    username = session.get("username","-")
    path = AUDIT_DIR / "audit.csv"
    new = not path.exists()
    with open(path, "a", newline="", encoding="utf-8") as f:
        w = csv.writer(f)
        if new: w.writerow(["timestamp","username","action","detail"])
        w.writerow([ts, username, action, detail])

def load_users():
    try: return json.loads(USERS_FILE.read_text(encoding="utf-8"))["users"]
    except Exception: return []

def save_users(users):
    USERS_FILE.write_text(json.dumps({"users":users}, indent=2), encoding="utf-8")

def load_items():
    try:
        items = json.loads(ITEMS_FILE.read_text(encoding="utf-8"))["items"]
        changed = False
        for it in items:
            for k, v in {"opening_stock":0.0,"min_stock":0.0,"barcode":""}.items():
                if k not in it:
                    it[k]=v; changed=True
        if changed: save_items(items)
        return items
    except Exception: return []

def save_items(items):
    ITEMS_FILE.write_text(json.dumps({"items":items}, indent=2), encoding="utf-8")

def now_local(): return datetime.now()

def hour_paths(ts):
    y,m,d,h = ts.strftime("%Y"), ts.strftime("%m"), ts.strftime("%d"), ts.strftime("%H")
    base_day_dir = DAILY_DIR / y / m / d
    base_hour_dir = base_day_dir / h
    base_day_dir.mkdir(parents=True, exist_ok=True)
    base_hour_dir.mkdir(parents=True, exist_ok=True)
    return base_hour_dir/"usage.csv", base_hour_dir/"usage.xlsx", base_day_dir/"usage_master.csv"

def append_log(record):
    ts = now_local()
    hourly_csv, hourly_xlsx, daily_csv = hour_paths(ts)
    is_new_hour = not hourly_csv.exists()
    is_new_day = not daily_csv.exists()
    headers = ["timestamp","username","item","qty","unit","notes","kind","shift","area","barcode"]
    row = [record.get("timestamp"), record.get("username"), record.get("item"), record.get("qty"),
           record.get("unit"), record.get("notes",""), record.get("kind"), record.get("shift",""),
           record.get("area",""), record.get("barcode","")]
    with open(hourly_csv,"a",newline="",encoding="utf-8") as f:
        w = csv.writer(f)
        if is_new_hour: w.writerow(headers)
        w.writerow(row)
    with open(daily_csv,"a",newline="",encoding="utf-8") as f:
        w = csv.writer(f)
        if is_new_day: w.writerow(headers)
        w.writerow(row)
    if pd is not None:
        try:
            df = pd.read_csv(hourly_csv)
            df.to_excel(hourly_xlsx, index=False)
        except Exception as e:
            print("Excel export failed:", e)

def iter_daily_csvs():
    if not DAILY_DIR.exists(): return
    for y in sorted([p for p in DAILY_DIR.iterdir() if p.is_dir()]):
        for m in sorted([p for p in y.iterdir() if p.is_dir()]):
            for d in sorted([p for p in m.iterdir() if p.is_dir()]):
                daily_csv = d/"usage_master.csv"
                if daily_csv.exists(): yield daily_csv

def load_all_logs(start_dt=None, end_dt=None):
    rows=[]
    for path in iter_daily_csvs():
        day = path.parent
        day_date = datetime.strptime(f"{day.parent.parent.name}-{day.parent.name}-{day.name}", "%Y-%m-%d")
        if start_dt and day_date.date()<start_dt.date(): continue
        if end_dt and day_date.date()>end_dt.date(): continue
        with open(path,"r",encoding="utf-8") as f:
            reader = csv.DictReader(f)
            for r in reader:
                rows.append(r)
    return rows

def compute_stock_snapshot():
    items = load_items()
    rows = load_all_logs()
    agg = {}
    for it in items:
        agg[it["name"]] = {"opening": float(it.get("opening_stock",0)), "in":0.0, "out":0.0, "unit":it["unit"], "min_stock": float(it.get("min_stock",0))}
    for r in rows:
        name = r["item"]; qty = float(r.get("qty",0) or 0); kind = (r.get("kind") or "OUT").upper()
        if name not in agg:
            agg[name]={"opening":0.0,"in":0.0,"out":0.0,"unit":r.get("unit",""),"min_stock":0.0}
        if kind=="IN": agg[name]["in"]+=qty
        else: agg[name]["out"]+=qty
    for name,v in agg.items():
        v["current"] = v["opening"] + v["in"] - v["out"]
    return agg

# ==== Reporting helpers ====
def load_all_logs_filtered(start_dt=None, end_dt=None, user=None, shift=None, area=None):
    rows=[]
    for path in iter_daily_csvs():
        day = path.parent
        day_date = datetime.strptime(f"{day.parent.parent.name}-{day.parent.name}-{day.name}", "%Y-%m-%d")
        if start_dt and day_date.date()<start_dt.date(): continue
        if end_dt and day_date.date()>end_dt.date(): continue
        with open(path,"r",encoding="utf-8") as f:
            reader = csv.DictReader(f)
            for r in reader:
                if user and r.get("username","")!=user: continue
                if shift and (r.get("shift","") or "").lower()!=shift.lower(): continue
                if area and (r.get("area","") or "").lower()!=area.lower(): continue
                rows.append(r)
    return rows

def summarize_rows(rows):
    by_item = {}
    totals = {"IN":0.0,"OUT":0.0}
    for r in rows:
        name = r["item"]; kind = (r.get("kind") or "OUT").upper()
        qty = float(r.get("qty",0) or 0); unit = r.get("unit","")
        if name not in by_item: by_item[name]={"unit":unit,"IN":0.0,"OUT":0.0}
        by_item[name][kind] += qty
        totals[kind] += qty
    result = []
    for name, v in by_item.items():
        result.append({"item":name,"unit":v["unit"],"in_qty":v["IN"],"out_qty":v["OUT"]})
    result.sort(key=lambda x: x["item"].lower())
    return result, totals

def iso_week_range(y, w):
    start = datetime.strptime(f"{y}-W{w}-1", "%G-W%V-%u")
    end = start + timedelta(days=6)
    return start, end

def month_range(ym):
    start = datetime.strptime(ym+"-01", "%Y-%m-%d")
    if start.month==12: nextm = datetime(start.year+1,1,1)
    else: nextm = datetime(start.year, start.month+1,1)
    end = nextm - timedelta(days=1)
    return start, end

def xlsx_from_summary(summary_rows, extra_cols=None, filename="report.xlsx"):
    if pd is None:
        out = io.StringIO(); w = csv.writer(out)
        headers = ["Item","Unit","Qty IN","Qty OUT"]
        if extra_cols: headers += list(extra_cols.keys())
        w.writerow(headers)
        for r in summary_rows:
            row = [r["item"], r["unit"], r["in_qty"], r["out_qty"]]
            if extra_cols: row += [extra_cols[k] for k in extra_cols]
            w.writerow(row)
        out.seek(0)
        return send_file(io.BytesIO(out.getvalue().encode("utf-8")), as_attachment=True, download_name=filename.replace(".xlsx",".csv"), mimetype="text/csv")
    df = pd.DataFrame(summary_rows)
    if extra_cols:
        for k,v in extra_cols.items():
            df[k] = v
    buf = io.BytesIO()
    with pd.ExcelWriter(buf, engine="openpyxl") as writer:
        df.to_excel(writer, index=False, sheet_name="Summary")
    buf.seek(0)
    return send_file(buf, as_attachment=True, download_name=filename, mimetype="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet")

# ==== Flask app ====
app = Flask(
    __name__,
    template_folder=str((BASE_DIR / "templates")),
    static_folder=str((BASE_DIR / "static"))
)
app.secret_key = os.environ.get("SECRET_KEY","dev-secret-key")
ensure_storage()

def current_user():
    if "username" in session:
        for u in load_users():
            if u["username"] == session["username"]:
                return u
    return None

def require_not_viewer():
    u = current_user()
    if u and u.get("role")=="viewer":
        flash("Akses dibatasi (viewer).","warning")
        return False
    return True

def login_required(fn):
    from functools import wraps
    @wraps(fn)
    def wrapper(*args, **kwargs):
        if not current_user(): return redirect(url_for("login"))
        return fn(*args, **kwargs)
    return wrapper

@app.route("/")
def index():
    if current_user(): return redirect(url_for("record"))
    return redirect(url_for("login"))

@app.route("/login", methods=["GET","POST"])
def login():
    if request.method=="POST":
        username = request.form.get("username","").strip()
        password = request.form.get("password","").strip()
        for u in load_users():
            if u["username"]==username and check_password_hash(u["password_hash"], password):
                session["username"]=username; log_audit("login","success"); flash("Login sukses","success")
                return redirect(url_for("record"))
        log_audit("login_fail", f"user={username}"); flash("Username atau password salah","danger")
    return render_template("login.html", title="Login | "+APP_TITLE)

@app.route("/logout")
def logout():
    log_audit("logout","user logged out")
    session.clear(); flash("Anda telah logout","info"); return redirect(url_for("login"))

# Simplified unified form for IN/OUT
@app.route("/record", methods=["GET","POST"])
@login_required
def record():
    items = load_items()
    if request.method=="POST":
        if not require_not_viewer(): return redirect(url_for("record"))
        kind = request.form.get("kind","OUT").upper()
        barcode = request.form.get("barcode","").strip()
        item_name = request.form.get("item")
        qty = request.form.get("qty","0")
        shift = request.form.get("shift","")
        area = request.form.get("area","")
        notes = request.form.get("notes","")
        try: qty_val = float(qty)
        except: flash("Qty harus angka","danger"); return redirect(url_for("record"))
        item = next((i for i in items if (barcode and i.get("barcode")==barcode) or i["name"]==item_name), None)
        if not item: flash("Item tidak ditemukan (cek barcode/nama)","danger"); return redirect(url_for("record"))
        timestamp = now_local().strftime("%Y-%m-%d %H:%M:%S")
        rec = {"timestamp":timestamp,"username":session["username"],"item":item["name"],"qty":qty_val,"unit":item["unit"],
               "notes":notes,"kind":kind,"shift":shift,"area":area,"barcode":item.get("barcode","")}
        append_log(rec); log_audit(f"{kind.lower()}_add", f"item={item['name']}, qty={qty_val}, shift={shift}, area={area}")
        flash(f"{kind} dicatat.","success"); 
        return redirect(url_for("record"))
    # today recent
    ts = now_local(); _,_, daily = hour_paths(ts)
    recent=[]
    if daily.exists():
        with open(daily,"r",encoding="utf-8") as f:
            reader = csv.DictReader(f)
            recent = list(reader)[-100:]
    stock = compute_stock_snapshot()
    return render_template("record.html", title="Record | "+APP_TITLE, items=items, recent=recent, stock=stock)

@app.route("/items", methods=["GET","POST"])
@login_required
def items():
    u = current_user()
    if u["role"] not in ("admin",): 
        flash("Hanya admin yang dapat mengelola item","warning"); 
        return redirect(url_for("record"))
    items = load_items()
    if request.method=="POST":
        action = request.form.get("action")
        if action=="add":
            name = request.form.get("name","").strip()
            unit = request.form.get("unit","").strip()
            opening_stock = float(request.form.get("opening_stock","0") or 0)
            min_stock = float(request.form.get("min_stock","0") or 0)
            barcode = request.form.get("barcode","").strip()
            if not name or not unit: flash("Nama & satuan wajib","danger")
            elif any(i["name"].lower()==name.lower() for i in items): flash("Item sudah ada","warning")
            else:
                items.append({"name":name,"unit":unit,"opening_stock":opening_stock,"min_stock":min_stock,"barcode":barcode})
                save_items(items); log_audit("item_add", name); flash("Item ditambahkan","success")
        elif action=="delete":
            name = request.form.get("name","").strip()
            items = [i for i in items if i["name"]!=name]
            save_items(items); log_audit("item_del", name); flash("Item dihapus","info")
        elif action=="update":
            name = request.form.get("name","").strip()
            opening_stock = float(request.form.get("opening_stock","0") or 0)
            min_stock = float(request.form.get("min_stock","0") or 0)
            unit = request.form.get("unit","").strip()
            barcode = request.form.get("barcode","").strip()
            for i in items:
                if i["name"]==name:
                    i["opening_stock"]=opening_stock; i["min_stock"]=min_stock
                    if unit: i["unit"]=unit
                    i["barcode"]=barcode
            save_items(items); log_audit("item_upd", name); flash("Item di-update","success")
        return redirect(url_for("items"))
    return render_template("items.html", title="Items | "+APP_TITLE, items=items)

@app.route("/users", methods=["GET","POST"])
@login_required
def users():
    u = current_user()
    if u["role"] not in ("admin",): 
        flash("Hanya admin yang dapat mengelola user","warning"); 
        return redirect(url_for("record"))
    users = load_users()
    if request.method=="POST":
        action = request.form.get("action")
        if action=="add":
            username = request.form.get("username","").strip()
            password = request.form.get("password","").strip()
            role = request.form.get("role","user")
            if not username or not password: flash("Username & password wajib","danger")
            elif any(x["username"].lower()==username.lower() for x in users): flash("Username sudah ada","warning")
            else:
                users.append({"username":username,"password_hash":generate_password_hash(password),"role":role})
                save_users(users); log_audit("user_add", username); flash("User ditambahkan","success")
        elif action=="delete":
            username = request.form.get("username","").strip()
            users = [x for x in users if x["username"]!=username]
            save_users(users); log_audit("user_del", username); flash("User dihapus","info")
        return redirect(url_for("users"))
    return render_template("users.html", title="Users | "+APP_TITLE, users=users)

# KPI
def summarize_period(start_dt, end_dt):
    rows = load_all_logs(start_dt, end_dt)
    per_item = {}
    per_user = {}
    per_area = {}
    for r in rows:
        item = r["item"]; user = r["username"]; area = r.get("area","")
        kind = (r.get("kind") or "OUT").upper(); qty = float(r.get("qty",0) or 0)
        if item not in per_item: per_item[item]={"IN":0.0,"OUT":0.0,"unit":r.get("unit","")}
        per_item[item][kind] = per_item[item].get(kind,0.0) + qty
        if user not in per_user: per_user[user]={"IN":0.0,"OUT":0.0}
        per_user[user][kind] = per_user[user].get(kind,0.0) + qty
        if area not in per_area: per_area[area]={"IN":0.0,"OUT":0.0}
        per_area[area][kind] = per_area[area].get(kind,0.0) + qty
    return rows, per_item, per_user, per_area

@app.route("/kpi")
@login_required
def kpi():
    today = date.today()
    iso_year, iso_week, _ = today.isocalendar()
    d = datetime.strptime(f"{iso_year}-W{iso_week}-1", "%G-W%V-%u")
    w_start, w_end = d, d+timedelta(days=6)
    w_rows, w_item, w_user, w_area = summarize_period(w_start, w_end)
    m_start = datetime(today.year, today.month, 1)
    m_end = (datetime(today.year+1,1,1) if today.month==12 else datetime(today.year, today.month+1,1)) - timedelta(days=1)
    m_rows, m_item, m_user, m_area = summarize_period(m_start, m_end)
    return render_template("kpi.html", title="KPI | "+APP_TITLE,
                           w_item=w_item, w_user=w_user, w_area=w_area,
                           m_item=m_item, m_user=m_user, m_area=m_area,
                           w_range=f"{w_start.date()} s.d. {w_end.date()}",
                           m_range=f"{m_start.date()} s.d. {m_end.date()}")

# Export ALL
@app.route("/export/all.xlsx")
@login_required
def export_all_xlsx():
    if pd is None:
        out = io.StringIO()
        w = csv.writer(out)
        w.writerow(["sheet","timestamp","username","item","qty","unit","notes","kind","shift","area","barcode"])
        for path in iter_daily_csvs():
            with open(path,"r",encoding="utf-8") as f:
                r = csv.reader(f); rows = list(r)
                for row in rows[1:]:
                    w.writerow(["logs"]+row)
        out.seek(0)
        log_audit("export_all_csv")
        return send_file(io.BytesIO(out.getvalue().encode("utf-8")), as_attachment=True, download_name="all_data.csv", mimetype="text/csv")
    buf = io.BytesIO()
    with pd.ExcelWriter(buf, engine="openpyxl") as writer:
        items = load_items()
        users = load_users()
        all_logs = []
        for path in iter_daily_csvs():
            df = pd.read_csv(path)
            all_logs.append(df)
        df_items = pd.DataFrame(items)
        df_users = pd.DataFrame([{"username":u["username"],"role":u["role"]} for u in users])
        df_logs = pd.concat(all_logs, ignore_index=True) if all_logs else pd.DataFrame()
        df_items.to_excel(writer, index=False, sheet_name="Items")
        df_users.to_excel(writer, index=False, sheet_name="Users")
        df_logs.to_excel(writer, index=False, sheet_name="Logs")
    buf.seek(0)
    log_audit("export_all_xlsx")
    return send_file(buf, as_attachment=True, download_name="all_data.xlsx", mimetype="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet")

# Backup Google Sheets
@app.route("/backup/google-sheets", methods=["POST"])
@login_required
def backup_google_sheets():
    if not require_not_viewer(): return redirect(url_for("record"))
    try:
        import gspread
        from google.oauth2.service_account import Credentials
    except Exception:
        flash("gspread/google-auth belum terpasang. Jalankan: pip install gspread google-auth","danger")
        return redirect(url_for("record"))
    sheet_key = os.environ.get("GSHEET_KEY") or request.form.get("sheet_key")
    cred_path = os.environ.get("GOOGLE_APPLICATION_CREDENTIALS") or request.form.get("cred_path")
    if not sheet_key or not cred_path:
        flash("Butuh GSHEET_KEY dan GOOGLE_APPLICATION_CREDENTIALS (path file JSON)","danger")
        return redirect(url_for("record"))
    scopes = ["https://www.googleapis.com/auth/spreadsheets"]
    creds = Credentials.from_service_account_file(cred_path, scopes=scopes)
    gc = gspread.authorize(creds)
    sh = gc.open_by_key(sheet_key)
    items = load_items()
    users = load_users()
    logs_rows = []
    headers = ["timestamp","username","item","qty","unit","notes","kind","shift","area","barcode"]
    for path in iter_daily_csvs():
        with open(path,"r",encoding="utf-8") as f:
            reader = csv.reader(f)
            for i, row in enumerate(reader):
                if i==0: 
                    headers = row
                    continue
                logs_rows.append(row)
    try:
        ws = sh.worksheet("Items"); ws.clear()
    except:
        ws = sh.add_worksheet(title="Items", rows=1000, cols=10)
    ws.update([list(items[0].keys())] + [list(i.values()) for i in items] if items else [["name","unit","opening_stock","min_stock","barcode"]])
    try:
        wu = sh.worksheet("Users"); wu.clear()
    except:
        wu = sh.add_worksheet(title="Users", rows=1000, cols=10)
    wu.update([["username","role"]] + [[u["username"],u["role"]] for u in users])
    try:
        wl = sh.worksheet("Logs"); wl.clear()
    except:
        wl = sh.add_worksheet(title="Logs", rows=2000, cols=12)
    wl.update([headers] + logs_rows if logs_rows else [["timestamp","username","item","qty","unit","notes","kind","shift","area","barcode"]])
    log_audit("backup_gsheet","ok")
    flash("Backup ke Google Sheets OK","success")
    return redirect(url_for("record"))

# === Reports & Comparison ===
@app.route("/reports/weekly")
@login_required
def reports_weekly():
    today = date.today()
    iso_year, iso_week, _ = today.isocalendar()
    year = int(request.args.get("year", iso_year))
    week = int(request.args.get("week", iso_week))
    user = request.args.get("user") or None
    shift = request.args.get("shift") or None
    area = request.args.get("area") or None
    start, end = iso_week_range(year, week)
    rows = load_all_logs_filtered(start, end, user, shift, area)
    summary, totals = summarize_rows(rows)
    return render_template("reports_weekly.html", title="Weekly Report | "+APP_TITLE, year=year, week=week, start=start.date(), end=end.date(), rows=summary, totals=totals, f_user=user, f_shift=shift, f_area=area)

@app.route("/reports/monthly")
@login_required
def reports_monthly():
    today = date.today()
    ym = request.args.get("ym", today.strftime("%Y-%m"))
    user = request.args.get("user") or None
    shift = request.args.get("shift") or None
    area = request.args.get("area") or None
    start, end = month_range(ym)
    rows = load_all_logs_filtered(start, end, user, shift, area)
    summary, totals = summarize_rows(rows)
    return render_template("reports_monthly.html", title="Monthly Report | "+APP_TITLE, ym=ym, start=start.date(), end=end.date(), rows=summary, totals=totals, f_user=user, f_shift=shift, f_area=area)

@app.route("/export/reports/weekly.xlsx")
@login_required
def export_reports_weekly_xlsx():
    today = date.today()
    iso_year, iso_week, _ = today.isocalendar()
    year = int(request.args.get("year", iso_year))
    week = int(request.args.get("week", iso_week))
    user = request.args.get("user") or None
    shift = request.args.get("shift") or None
    area = request.args.get("area") or None
    start, end = iso_week_range(year, week)
    rows = load_all_logs_filtered(start, end, user, shift, area)
    summary, totals = summarize_rows(rows)
    extras = {"Total IN": totals["IN"], "Total OUT": totals["OUT"]}
    return xlsx_from_summary(summary, extras, filename=f"weekly_{year}_W{week}.xlsx")

@app.route("/export/reports/monthly.xlsx")
@login_required
def export_reports_monthly_xlsx():
    today = date.today()
    ym = request.args.get("ym", today.strftime("%Y-%m"))
    user = request.args.get("user") or None
    shift = request.args.get("shift") or None
    area = request.args.get("area") or None
    start, end = month_range(ym)
    rows = load_all_logs_filtered(start, end, user, shift, area)
    summary, totals = summarize_rows(rows)
    extras = {"Total IN": totals["IN"], "Total OUT": totals["OUT"]}
    return xlsx_from_summary(summary, extras, filename=f"monthly_{ym}.xlsx")

@app.route("/compare")
@login_required
def compare_periods():
    today = date.today()
    iso_year, iso_week, _ = today.isocalendar()
    this_w_start, this_w_end = iso_week_range(iso_year, iso_week)
    last_week_date = this_w_start - timedelta(days=7)
    last_y, last_w, _ = last_week_date.isocalendar()
    last_w_start, last_w_end = iso_week_range(last_y, last_w)
    this_m_ym = today.strftime("%Y-%m")
    this_m_start, this_m_end = month_range(this_m_ym)
    if this_m_start.month==1: last_m_ym = f"{this_m_start.year-1}-12"
    else: last_m_ym = f"{this_m_start.year}-{this_m_start.month-1:02d}"
    last_m_start, last_m_end = month_range(last_m_ym)
    w_this_rows = load_all_logs_filtered(this_w_start, this_w_end)
    w_last_rows = load_all_logs_filtered(last_w_start, last_w_end)
    w_this_sum, w_this_tot = summarize_rows(w_this_rows)
    w_last_sum, w_last_tot = summarize_rows(w_last_rows)
    m_this_rows = load_all_logs_filtered(this_m_start, this_m_end)
    m_last_rows = load_all_logs_filtered(last_m_start, last_m_end)
    m_this_sum, m_this_tot = summarize_rows(m_this_rows)
    m_last_sum, m_last_tot = summarize_rows(m_last_rows)
    def total_dict(t): return {"IN": t["IN"], "OUT": t["OUT"], "NET": t["IN"]-t["OUT"]}
    return render_template("compare.html", title="Weekly/Monthly Comparison | "+APP_TITLE,
                           w_this=total_dict(w_this_tot), w_last=total_dict(w_last_tot),
                           m_this=total_dict(m_this_tot), m_last=total_dict(m_last_tot),
                           w_label=f"{this_w_start.date()} s.d. {this_w_end.date()}",
                           w_prev_label=f"{last_w_start.date()} s.d. {last_w_end.date()}",
                           m_label=f"{this_m_start.date()} s.d. {this_m_end.date()}",
                           m_prev_label=f"{last_m_start.date()} s.d. {last_m_end.date()}")

@app.route("/export/compare.xlsx")
@login_required
def export_compare_xlsx():
    if pd is None:
        out = io.StringIO(); w = csv.writer(out)
        w.writerow(["Period","IN","OUT","NET"])
        today = date.today()
        iso_year, iso_week, _ = today.isocalendar()
        this_w_start, this_w_end = iso_week_range(iso_year, iso_week)
        last_week_date = this_w_start - timedelta(days=7)
        last_y, last_w, _ = last_week_date.isocalendar()
        last_w_start, last_w_end = iso_week_range(last_y, last_w)
        def totals(start,end):
            _, t = summarize_rows(load_all_logs_filtered(start,end))
            return [t["IN"], t["OUT"], t["IN"]-t["OUT"]]
        w.writerow([f"This Week ({this_w_start.date()}-{this_w_end.date()})"] + totals(this_w_start,this_w_end))
        w.writerow([f"Last Week ({last_w_start.date()}-{last_w_end.date()})"] + totals(last_w_start,last_w_end))
        this_m_ym = today.strftime("%Y-%m")
        this_m_start, this_m_end = month_range(this_m_ym)
        if this_m_start.month==1: last_m_ym = f"{this_m_start.year-1}-12"
        else: last_m_ym = f"{this_m_start.year}-{this_m_start.month-1:02d}"
        last_m_start, last_m_end = month_range(last_m_ym)
        w.writerow([f"This Month ({this_m_start.date()}-{this_m_end.date()})"] + totals(this_m_start,this_m_end))
        w.writerow([f"Last Month ({last_m_start.date()}-{last_m_end.date()})"] + totals(last_m_start,last_m_end))
        out.seek(0)
        return send_file(io.BytesIO(out.getvalue().encode("utf-8")), as_attachment=True, download_name="compare.csv", mimetype="text/csv")
    import pandas as pd
    buf = io.BytesIO()
    with pd.ExcelWriter(buf, engine="openpyxl") as writer:
        def make_df(title, start, end):
            _, t = summarize_rows(load_all_logs_filtered(start,end))
            return pd.DataFrame([{"Period": title, "IN": t["IN"], "OUT": t["OUT"], "NET": t["IN"]-t["OUT"]}])
        today = date.today()
        iso_year, iso_week, _ = today.isocalendar()
        this_w_start, this_w_end = iso_week_range(iso_year, iso_week)
        last_week_date = this_w_start - timedelta(days=7)
        last_y, last_w, _ = last_week_date.isocalendar()
        last_w_start, last_w_end = iso_week_range(last_y, last_w)
        dfw = pd.concat([
            make_df("This Week", this_w_start, this_w_end),
            make_df("Last Week", last_w_start, last_w_end),
        ], ignore_index=True)
        this_m_ym = today.strftime("%Y-%m")
        this_m_start, this_m_end = month_range(this_m_ym)
        if this_m_start.month==1: last_m_ym = f"{this_m_start.year-1}-12"
        else: last_m_ym = f"{this_m_start.year}-{this_m_start.month-1:02d}"
        last_m_start, last_m_end = month_range(last_m_ym)
        dfm = pd.concat([
            make_df("This Month", this_m_start, this_m_end),
            make_df("Last Month", last_m_start, last_m_end),
        ], ignore_index=True)
        dfw.to_excel(writer, index=False, sheet_name="Weekly")
        dfm.to_excel(writer, index=False, sheet_name="Monthly")
    buf.seek(0)
    return send_file(buf, as_attachment=True, download_name="compare.xlsx", mimetype="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet")

if __name__ == "__main__":
    ensure_storage()
    app.run(host="0.0.0.0", port=5000, debug=True)
